/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* Alpha Diagnostics Memory Management Framework */
/* $Id: memory.c,v 1.8 2001/03/09 19:30:30 stig Exp $
 * $Revision: 1.8 $
 * $Author: stig $
 */

#undef TRACE_ENABLE	/* NB: be aware of complications */
			/* NB: the BIOS emulator makes millions of mallocs */

#undef MEM_PARANOIA	/* define this to enable thorough sanity checks */
			/* Note: currently BREAKS the DBLX file processing! */

/*-------------------------------------------------------------------*/
#undef DEBUG_HARNESS

#ifdef DEBUG_HARNESS	/* DEBUG_HARNESS */

#include <stdlib.h>
#include <stdio.h>

#include "include/memory.h"

#ifdef TRACE_ENABLE

#define TRACE(format, args...) \
        printf( __FUNCTION__ ": " format , ## args)
#else

#define TRACE( format, args... )

#endif  /* TRACE_ENABLE */


#define FALSE 0
#define TRUE 1

/* A handy assertion-style macro for testing for internally-caused errors */

#define BUGCHECK( expr )                                                \
    {                                                                   \
        if( expr )      printf( "BUGCHECK file %s, line %d\n",		\
						__FILE__, __LINE__ );  \
    }

typedef unsigned char uint8;
typedef unsigned long uint64;



#else			/* DEBUG_HARNESS */
/*-------------------------------------------------------------------*/

#include "lib.h"
#include "specifics.h"
#include "hwrpb.h"
#include "uilib.h"
#include "platform.h"

#endif			/* DEBUG_HARNESS */


static BOOLEAN mem_init_done = FALSE;

/*----------------------------------------------------------------------*/
/* Page allocation management */
/* The page is the basic unit of memory.  The PALcode, Diags and kernel 
 * may all reserve different pages in memory via the HWRPB memory map
 * at various stages of operation.  Adherence to who has reserved what is
 * not enforced so must be taken into consideration whenever using a new 
 * memory region.
 */

typedef uint8 usage_t;

static usage_t *mem_map = NULL;
static unsigned map_pages = 0;

/* Request a number of contiguous pages, returning base PFN (or -1) */
int page_alloc( int n_pfns, uint8 usage )
{
    int base, i;

    TRACE( "Running, looking for %d pages, usage %d\n", n_pfns, usage );

    for ( base = 0; base < (map_pages - n_pfns); base++ )
    {
	for ( i=0; i<n_pfns; i++ )
	{
	    if ( mem_map[ base + i ] != FREE_PAGE )
		break;
	}

	/* Did we have a match? */
	if( i==n_pfns )
	{
	    TRACE( "Match found at PFN %d (%dKB)\n",
			base, base << (PAGE_SHIFT-10) );
	    page_mark_range( base, n_pfns, usage );
	    return base;
	}

	/* Otherwise move on by the amount scanned and start afresh */
	base += i;
    }

    /* There is not enough contiguous space to satisfy the alloc - uh-oh... */
    return -1;
}


/* Mark a particular region on the memory map */
void page_mark_range( int base_pfn, int n_pfns, uint8 usage )
{
    TRACE( "marking %d pages at page %d, usage %d\n", n_pfns, base_pfn, usage );

    while ( n_pfns != 0 )
    {
	/* We should not have the situation where a page goes from x -> y where
	 * neither x nor y are FREE_PAGE (means one alloc over another one) */
	if ( mem_map[ base_pfn ] != FREE_PAGE && usage != FREE_PAGE &&
	     mem_map[ base_pfn ] != usage )
	{
	    mobo_logf( LOG_WARN "MEM: page %d being changed from %d->%d\n",
			base_pfn, mem_map[ base_pfn ], usage );
	}

        mem_map[ base_pfn ] = usage;
        base_pfn++, n_pfns--;
    }
    hwrpb_update_mem( mem_map, map_pages );
}


/* Find the largest possible contiguous allocation of memory */
/* Useful for the memory tests */
void page_max_free( int *base_pfn, int *n_pfns )
{
    int bestbase, bestsize;
    int base, i;

    for ( base=0, bestbase=0, bestsize=0; base < map_pages; base++ )
    {
	for ( i=0; i < (map_pages-base); i++ )
	{
	    if ( mem_map[ base + i ] != FREE_PAGE )
		break;
	}

	/* Did we have a match? */
	if ( i > bestsize )
	{
	    bestbase = base, bestsize = i;
	}

	/* Otherwise move on by the amount scanned and start afresh */
	base += i;
    }

    /* hand the biggest allocation info back to the caller */
    *base_pfn = bestbase, *n_pfns = bestsize;
}



/*----------------------------------------------------------------------*/
/* Heap allocation: data structures and basic management routines */

/* All addresses returned are aligned on this boundary, which must be equal
 * or larger than sizeof( alloc_t ) */

#define ALLOC_GRANULARITY	16

typedef struct _alloc_t {
    uint64 size;
    struct _alloc_t *next;
} alloc_t;


/* The free list starts at the bottom of the heap and is an ordered list of 
 * increasing addresses of free blocks.
 */
static alloc_t *free_list = NULL;

/* The alloc list is similar to the free list but is an ordered index 
 * of allocated blocks.
 */
static alloc_t *alloc_list = NULL;


/* List management routines.  The alloc and free lists are large linked lists,
 * the only special feature being that the pointers used have greater 
 * significance for the data they describe. */

static alloc_t *add_to_list( alloc_t *list, alloc_t *A )
{
    alloc_t *L;

    TRACE( "Running on alloc at 0x%lX (size %d)\n", A, A->size );

    /* First possibility - if the list is NULL we simply create it with
     * this handle. */
    if ( list == NULL )
    {
	TRACE( "List was NULL, starting new list\n" );
	A->next = NULL;
	return A;
    }


    /* Second possibility - the handle being added is less than the current
     * top of the list.  So, we have a new start for the list.
     * (Handles are kept sorted by address)
     */
    if ( A < list )
    {
	TRACE( "Creating new start of list\n" );
	A->next = list;
	return A;
    }

    /* Third possibility - the handle being freed is somewhere in the middle
     * of the list.  From previous tests the list is not NULL.
     */
    L = list;
    while( L->next != NULL )
    {
	if ( L->next > A )
	{
	    A->next = L->next;
	    L->next = A;
	    TRACE( "Inserting after 0x%lX (next 0x%lX)\n", L, A->next );
	    return list;
	}
	L = L->next;			/* on to next free list handle */
    }

    /* Fourth possibility - the handle being freed is at the top of the heap,
     * beyond the end of the current free list
     */
    TRACE( "Appending to end of list\n", L, A->next );
    A->next = NULL;
    L->next = A;
    return list;
}


/* Remove a handle from the list - for example from the free list 
 * for a heap allocation, or from the allocated list for a free(). */

static alloc_t *remove_from_list( alloc_t *list, alloc_t *A )
{
    alloc_t *L;

    TRACE( "List at 0x%lX handle at 0x%lX\n", list, A );

    /* What if the list is NULL (ie, nothing to remove)
     * In which case, we can't be allocating from the list
     * - internal inconsistency!
     */
    BUGCHECK( list == NULL );

    /* Do we need to remove the head of list */
    if ( list == A )
    {
	return list->next;
    }

    /* Or is the handle is further into the list */
    L = list;
    while( L->next != NULL )
    { 
	if ( L->next == A )
	{
	    L->next = L->next->next;
	    return list;
	}
	L = L->next;			/* On to next handle */
    }

    /* Final possibility - alloc request for something not in the list */
    /* Only internal consistency could bring us here */
    BUGCHECK( TRUE );
    return NULL;			/* [keeps compiler happy] */
}


/* merge up to three successive list entries (one or more of which may be NULL)
 * to produce a single handle for a contiguous free region */
static void canonicalize( alloc_t *L )
{
    char *end, *start;

    static alloc_t *merge_check( alloc_t *L )
    {
	end = (char *)L + L->size + ALLOC_GRANULARITY;
	start = (char *)L->next;

	if ( end >= start )
	{
	    /* Subsume the second handle into the first */
	    L->size += ALLOC_GRANULARITY + L->next->size;
	    L->next = L->next->next;
	    return L;
	}
	else
	{
	    return L->next;
	}
    }


    if ( L==NULL )
	return;

    if ( L->next == NULL )
	return;

    /* Check for merging possibilities between first and second members */
    L = merge_check( L );

    if ( L->next == NULL )
	return;

    /* Look at the merging possibilities between middle and last members */
    L = merge_check( L );
}


/* Debug: dump our free blocks */
void heap_dump( void )
{
    alloc_t *L;
    int allocs = 0, alloc_bytes = 0, total_bytes = 0;

    TRACE( "Free list:\n" );
    L = free_list;
    while( L != NULL )
    {
	TRACE( " 0x%016lX: %d bytes\n", L, L->size );
	total_bytes += L->size + ALLOC_GRANULARITY;
	L = L->next;
    }

    TRACE( "Alloc list:\n" );
    L = alloc_list;
    while( L != NULL )
    {
	TRACE( " 0x%016lX: %d bytes\n", L, L->size );
	allocs++;
	alloc_bytes += L->size;
	total_bytes += L->size + ALLOC_GRANULARITY;
	L = L->next;
    }

    TRACE( "%d allocs using %d bytes (%d total = %d%% usage)\n",
	allocs, alloc_bytes, total_bytes, (alloc_bytes * 100) / total_bytes );
}




/*------------------------------------------------------------------------*/
/* Heap allocation: mid-level access */


/* Paranoia: Checking an address (heap alloc_t or raw data handle) is valid and 
 * from reserved heap memory. */

BOOLEAN check_address_on_heap( void *ptr )
{
    usage_t usage;
    uint64 addr = (uint64)ptr & ~DBM_SUPER;

    /* Check address is within physical memory */
    if ( addr >= primary_impure->MEM_SIZE )
    {
	mobo_logf( LOG_CRIT
	    "MEM: address is bad (outsize address 0x%016lX)\n", addr );
	return FALSE;
    }

    /* Check page is marked out for heap usage */
    usage = mem_map[ ADDR2PFN( addr, DBM_SUPER ) ];
    if ( usage != HEAP_PAGE )
    {
	mobo_logf( LOG_CRIT
	    "MEM: address beyond heapspace (address 0x%016lX marked %d)\n",
		    addr, usage );
	return FALSE;
    }

    /* Check alignment */
    if ( (addr % ALLOC_GRANULARITY) != 0 )
    {
	mobo_logf( LOG_CRIT
	    "MEM: address is bad (unaligned address 0x%016lX)\n", addr );
	return FALSE;
    }

    /* Address must be OK */
    return TRUE;
}


/* Paranoia: Check for free list consistency */

BOOLEAN free_list_consistency( void )
{
    alloc_t *tmp = free_list;

    /* An empty free list is OK */
    if ( free_list == NULL )
	return TRUE;

    if ( check_address_on_heap( tmp ) != TRUE )
    {
	mobo_logf( LOG_CRIT "MEM: free list is corrupt at its head!\n" );
	return FALSE;
    }

    while( tmp->next != NULL )
    {
	if ( check_address_on_heap( tmp->next ) != TRUE )
	{
	    mobo_logf( LOG_CRIT "MEM: free list is corrupt at addr 0x%016lX!\n",
		tmp );
	    return FALSE;
	}

	tmp = tmp->next;
    }

    /* Heap appears OK */
    return TRUE;
}


/* A full check of heap consistency */
void heap_valid( void )
{
    int pfn, npfns;
    void *block_base = NULL, *end;
    alloc_t *A, *F, *current, *next;

    /* If heap memory has not yet been configured we can't do anything yet */
    if( !mem_init_done )
	return;


    /* Pick up our alloc and free list pointers.  Everything in the heap must
     * be referenced by exactly one of these pointers. */

    A = alloc_list, F = free_list;
    TRACE( "alloc head = 0x%lX, free head = 0x%lX\n", A, F );

    pfn = 0;
    do
    {
	/* Find the next block of heap pages */
	for( ; pfn < map_pages; pfn++ )
	{
	    if ( mem_map[pfn] == HEAP_PAGE )
	    {
		block_base = PFN2ADDR( pfn, DBM_SUPER );
		break;
	    }
	}
	if ( pfn == map_pages )		/* Run out of heap? */
	    break;


	/* How many pages in this block? */
	for( npfns=0; pfn + npfns < map_pages; npfns++ )
	{
	    if ( mem_map[ pfn + npfns ] != HEAP_PAGE )
		break;
	}


	/* We have a contiguous block of heap memory for consistency check */
	current = block_base;

	/* pointer to first byte of memory beyond this block */
	end = PFN2ADDR( pfn + npfns, DBM_SUPER );

	do {
	    /* Validation on the current alloc data */
	    if( current->size % ALLOC_GRANULARITY != 0 )
	    {
		/* Alloc size inconsistency */
		mobo_logf( LOG_CRIT "MEM: heap inconsistency:\n"
		    LOG_CRIT "Alloc at 0x%lX, size %ld, next 0x%lX is bad\n"
		    LOG_CRIT "The size does not match granularity req'ments\n",
		    current, current->size, current->next );
		BUGCHECK( TRUE );
	    }

	    next = (alloc_t *)
			((char *)current + current->size + ALLOC_GRANULARITY);
	    if ( (void *)next > end )
	    {
		/* Alloc overflows block! */
		mobo_logf( LOG_CRIT "MEM: heap inconsistency:\n"
		    LOG_CRIT "Handle at 0x%lX, size %ld, next 0x%lX is bad\n"
		    LOG_CRIT "It runs beyond mapped heap space. (0x%lX)\n",
		    current, current->size, current->next, end );
		BUGCHECK( TRUE );
	    }

	    /* Which list owns the current pointer? */
	    if ( current != A && current != F )
	    {
		/* Orphaned alloc! */
		mobo_logf( LOG_CRIT "MEM: heap inconsistency:\n"
		    LOG_CRIT "Handle at 0x%lX, size %ld, next 0x%lX is bad\n"
		    LOG_CRIT "It is an orphan - not on alloc or free lists.\n",
		    current, current->size, current->next );
		BUGCHECK( TRUE );
	    }

	    /* Does it exist on both lists? */
	    if( A == F )
	    {
		mobo_logf( LOG_CRIT "MEM: heap inconsistency:\n"
                    LOG_CRIT "Handle at 0x%lX, size %ld, next 0x%lX is bad\n"
                    LOG_CRIT "It exists on both alloc and free lists!\n",
                    A, A->size, A->next );
                BUGCHECK( TRUE );

	    }

	    if ( current == A )		/* Current is allocated */
	    {
		if ( A->next != NULL && !check_address_on_heap( A->next ) )
		{
		    /* Alloc list is corrupted! */
		    mobo_logf( LOG_CRIT "MEM: heap inconsistency:\n"
			LOG_CRIT "Handle list has been corrupted: \n"
			LOG_CRIT "handle at 0x%lX (%d bytes) points to 0x%lX\n",
			A, A->size, A->next );
		    BUGCHECK( TRUE );
		}
		A = A->next;
	    }
	    else			/* Current is free */
	    {
		if ( F->next != NULL && !check_address_on_heap( F ) )
		{
		    /* Free list is corrupted! */
		    mobo_logf( LOG_CRIT "MEM: heap inconsistency:\n"
			LOG_CRIT "Free list has been corrupted: \n"
			LOG_CRIT "handle at 0x%lX (%d bytes) points to 0x%lX\n",
			F, F->size, F->next );
		    BUGCHECK( TRUE );
		}
		F = F->next;
	    }

	    /* Move on to next immediate allocation */
	    current = next;

	} while( (void *)current < end );

	pfn += npfns;

    } while( TRUE );

    /* Check that we have expended all our alloc and free lists */
    if( A != NULL )
    {
	mobo_logf( LOG_CRIT "MEM: heap inconsistency:\n"
	    LOG_CRIT "Alloc list has been corrupted: \n"
	    LOG_CRIT "Extra handles beginning at 0x%lX (%d bytes)\n",
	    A, A->size );
	BUGCHECK( TRUE );
    }
    if( F != NULL )
    {
	mobo_logf( LOG_CRIT "MEM: heap inconsistency:\n"
	    LOG_CRIT "Free list has been corrupted: \n"
	    LOG_CRIT "Extra handles beginning at 0x%lX (%d bytes)\n",
	    F, F->size );
	BUGCHECK( TRUE );
    }

    TRACE( "Heap is OK.\n" );
}







/* Scan the free list for an alloc that is big enough to meet requirements */
/* Claim a chunk of memory starting at the alloc handle */
/* Returns a pointer to the alloc_t handle of the memory region in question */
static alloc_t *handle_alloc( size_t nob )
{
    alloc_t *alloc, *remainder;
#ifdef MEM_PARANOIA
#define POISON_MAGIC	0xFACECAFEFACECAFEUL
    size_t bytes;
    uint64 *poison;
#endif

    TRACE( "Running on request for %d bytes\n", nob );


    alloc = free_list;
    while ( alloc != NULL )
    {
	if( alloc->size >= nob )
	    break;

	alloc = alloc->next;
    }

    if ( alloc == NULL )
    {
	/* there is nothing big enough for the request */
	TRACE( "Heap not big enough for allocation\n" );
	return NULL;
    }


    TRACE( "Running on alloc at 0x%016lX, %d bytes\n", alloc, nob );
    free_list = remove_from_list( free_list, alloc );

    /* Round up the allocation to the nearest block granularity */
    /* Here we assume that the alloc granularity is a power of two */
    if ( (nob & (ALLOC_GRANULARITY-1)) != 0 )
    {
	nob = (nob | (ALLOC_GRANULARITY-1)) + 1;
    }

    /* Is this allocation bigger than the requested size: recycle some? */
    /* Note - this may produce a zero-length handle for canonicalisation */
    if ( alloc->size > nob )
    {
	remainder = (alloc_t *)( (char *)alloc + nob + ALLOC_GRANULARITY );
	remainder->size = alloc->size - (nob + ALLOC_GRANULARITY);
	remainder->next = NULL;				/* defensive coding */
	free_list = add_to_list( free_list, remainder );
	canonicalize( remainder );	/* merge with next entry if poss. */
    }

    /* Claim the correct number of used bytes for this handle. */
    alloc->size = nob;
    alloc->next = NULL;		/* defensive */

#ifdef MEM_PARANOIA		/* MEM_PARANOIA */
    /* "Poison" the page with characteristic nasty data, in case it gets 
     * some nasty buffer overrun, in which case the poison might show up in
     * registers, etc.
     */

    poison = (uint64 *)((char *)alloc + ALLOC_GRANULARITY);
    for(  bytes = 0; bytes < nob; bytes += sizeof( *poison ) )
    {
	*poison++ = POISON_MAGIC;
    }
#endif				/* MEM_PARANOIA */

    alloc_list = add_to_list( alloc_list, alloc );
    return alloc;
}




/*----------------------------------------------------------------------*/
/* malloc / realloc / free standard heap interface */


/* Add a region of memory to the heap space (in units of pages) */
static void heap_add( int pool_base_pfn, int pool_pfns )
{
    size_t pool_size = pool_pfns * PAGE_SIZE;
    alloc_t *A = (alloc_t *)PFN2ADDR( pool_base_pfn, DBM_SUPER );

    TRACE("Setting up for base 0x%016lX, size %d\n", A, pool_size );

    A->size = pool_size - ALLOC_GRANULARITY;
    A->next = NULL;

    free_list = add_to_list( free_list, A );
    canonicalize( A );			/* merge with next entry if poss. */
}


static struct
{
    int malloc;
    int free;

} calls = { 0, 0 };


void *malloc( size_t size )
{
    alloc_t *L;
    int pfn, n_pfns;
    void *alloc;

#ifdef MEM_PARANOIA
    BUGCHECK( !smp_primary() );		/* FIXME BODGE DEBUG: Paranoia */

    /* Paranoia */
    heap_valid();
#endif

    if( !mem_init_done )
	mem_init();

    TRACE( "Running for %d bytes\n", size );

    calls.malloc++;

#ifdef TRACE_ENABLE
    heap_dump();
    hwrpb_dump_memcfg();
#endif

    L = handle_alloc( size );
    if ( L == NULL )
    {
	/* We need to grow the heap to make space for this allocation */
	/* Bump up the allocation size to factor in the tag structure */
	n_pfns = PFNCOUNT( size + ALLOC_GRANULARITY );
	pfn = page_alloc( n_pfns, HEAP_PAGE );
	TRACE( "Growing heap space with request for %d pages at %d\n",
		n_pfns, pfn );
	heap_add( pfn, n_pfns );

	L = handle_alloc( size );
	if ( L == NULL )
	{
	    /* Shouldn't happen - we know the heap is big enough */
	    mobo_logf( LOG_CRIT "MALLOC: failed on alloc of %d bytes!\n",
		size );
	    BUGCHECK( TRUE );
	}
    }

    alloc = (void *) ( (char *)L + ALLOC_GRANULARITY );
#ifdef MEM_PARANOIA
    heap_valid();
#endif
    TRACE( "Done: Alloc is 0x%lX\n", alloc );
    return alloc;
}


void free( void *ptr )
{
    alloc_t *L;

#ifdef MEM_PARANOIA
    BUGCHECK( !smp_primary() );		/* FIXME BODGE DEBUG: Paranoia */

    /* Paranoia */
    heap_valid();
#endif

    if( !mem_init_done )
	mem_init();

    TRACE( "Running for 0x%016lX\n", ptr );
    calls.free++;

    if ( check_address_on_heap( ptr ) == FALSE )
    {
	mobo_logf( LOG_CRIT
	    "MEM: attempt to free non-heap-space address 0x%016lX\n", ptr );
	BUGCHECK( TRUE );
    }

    L = (alloc_t *)( (char *)ptr - ALLOC_GRANULARITY );
    alloc_list = remove_from_list( alloc_list, L );
    free_list = add_to_list( free_list, L );
    canonicalize( L );			/* merge with next entry if poss. */
}


void *realloc( void *ptr, size_t size )
{
    void *newptr;

#ifdef MEM_PARANOIA
    BUGCHECK( !smp_primary() );		/* FIXME BODGE DEBUG: Paranoia */
#endif

    if( !mem_init_done )
	mem_init();

    free( ptr );
    newptr = malloc( size );
    if ( newptr != ptr )
    {
	/* NB copies some junk if bigger (OK, if we're not at top of mem) */
	memcpy( newptr, ptr, size );
    }
    return newptr;
}



/*----------------------------------------------------------------------*/
/* Memory management in general */

/* High-level routine, intended to be called once during initialisation,
 * which configures the page usage data structures and sets up a heap.
 */

void mem_init(void)
{
    unsigned long img_pages;
    unsigned long stack_pages;
    int heap_pages;

    if ( mem_init_done )
	return;

    if ( !smp_primary() )
	return;

    TRACE( "Running...\n" );

    /* Setup of memory map: the memory map is at a fixed location, defined 
     * in include/memory.h.  This is a bad idea because it may be true that
     * somebody has reserved that region of memory for other things (eg, 
     * reserved by a halting kernel for crashdump data)
     */

    mem_map = (char *)DIAGS_MEM_MAP;
    map_pages = PFNCOUNT( primary_impure->MEM_SIZE );

    /* Start with a clean slate, in a way that preserves HWRPB reservations */
    memset( mem_map, FREE_PAGE, map_pages );


    /* We'll need to import any reservations from the HWRPB, if there is one */
    hwrpb_update_mem_from_hwrpb();


    /* Reserved any fixed regions of memory.  We reserve everything
     * that we know about as a fixed region.  This is because if we don't 
     * reserve now, we may find that the heap as crept into that space
     */

    /* Reserve the page usage map itself.  Assume that its 1 byte per page */
    page_mark_range( ADDR2PFN( RAW_MEM_MAP, 0 ), PFNCOUNT( map_pages ),
			TEMPORARY_PAGE );

    /* PALcode and PAL data regions (nb here we assume PALcode <= 32K) */
    page_mark_range( ADDR2PFN( RAW_PAL_DATA, 0 ),
			PAL_DATA_PAGES, ALLOCATED_PAGE );
    page_mark_range( ADDR2PFN(primary_impure->PAL_BASE, DBM_SUPER),
			PAL_PAGES, ALLOCATED_PAGE );

    /* Reserve the diags compressed image */
    img_pages = PFNCOUNT( compressed_img_end - compressed_img_start );
    page_mark_range( ADDR2PFN(compressed_img_start, DBM_SUPER),
			img_pages, ALLOCATED_PAGE );

    /* Reserve the diags uncompressed image (nb this gets discarded on boot) */
    img_pages = PFNCOUNT( &_uncompressed_end - &_uncompressed_start );
    page_mark_range( ADDR2PFN(&_uncompressed_start, DBM_SUPER),
			img_pages, TEMPORARY_PAGE );

    /* Reserve the HWRPB page */
    page_mark_range( ADDR2PFN(hwrpb, DBM_SUPER), HWRPB_PAGES, ALLOCATED_PAGE );


    /* Here we reserve some stack for each processor */
    stack_pages = DIAGS_STACK_PAGES * MAX_CPUS;
    page_mark_range( ADDR2PFN( DIAGS_STACK, DBM_SUPER ) - stack_pages,
			stack_pages, TEMPORARY_PAGE );


    /* Reserve the Linux PALcode and kernel location.  Note: with 2.4-series
     * kernels, the base kernel address was moved to 8MB from 3MB.  We're only
     * reserving one and the lower addresses are more vulnerable to heap creep.
     * We reserve defaults as temporary pages to claim them here, but the linux 
     * loader must declare the correct addresses as allocated pages later.
     */
    page_mark_range( ADDR2PFN( RAW_OSFPAL, 0 ), PAL_PAGES, TEMPORARY_PAGE );
    page_mark_range( ADDR2PFN( RAW_KRNL, 0 ), LINUX_KRNL_PAGES, TEMPORARY_PAGE); 

#ifdef USE_DLD

    /* Reserve the applet download region */
    page_mark_range( ADDR2PFN( RAW_DLD, 0), APPLET_PAGES, TEMPORARY_PAGE );

#endif /* USE_DLD */



    /* Setup a starter size for the heap (or it may get rather fragmented) */
    TRACE( "Configuring %dKB heap memory\n",
		(POOL_DFLT_PAGES * PAGE_SIZE) >> 10 );

    heap_pages = page_alloc( POOL_DFLT_PAGES, HEAP_PAGE );

    if ( heap_pages != -1 )
	heap_add( heap_pages, POOL_DFLT_PAGES );

    TRACE( "Done.\n" );
    mem_init_done = TRUE;
}


